<?php
/**
 * Helper functions for theme templates
 *
 * @package Base_Theme
 */

/**
 * Get global button URL from theme settings
 *
 * @param string $default Default URL to return if not set in options
 * @return string URL for buttons
 */
function casino_get_global_button_url($default = '#') {
    $lang_slug = pll_current_language();
    $post_id = $lang_slug ? 'options_' . $lang_slug : 'option';
    $field_name = 'global_button_url_' . $lang_slug;
    $url = get_field($field_name, $post_id);
    return !empty($url) ? $url : $default;
}

/**
 * Get global button text (hardcoded function for backward compatibility)
 *
 * @param string $default Default text to return
 * @return string Button text
 */
function casino_get_global_button_text($default = 'Sign Up') {
    return $default;
}

/**
 * Get button HTML with proper text and URL
 *
 * @param array $args Button arguments
 * @return string Button HTML
 */
function casino_get_button($args = []) {
    $defaults = [
        'url' => casino_get_global_button_url(),
        'text' => casino_get_global_button_text(),
        'class' => 'button button--register button--colored',
        'target' => '',
        'data_attr' => [],
    ];

    $settings = wp_parse_args($args, $defaults);
    
    $attr_html = '';
    
    // Add data attributes
    if (!empty($settings['data_attr']) && is_array($settings['data_attr'])) {
        foreach ($settings['data_attr'] as $key => $value) {
            $attr_html .= ' data-' . esc_attr($key) . '="' . esc_attr($value) . '"';
        }
    }
    
    // Build target attribute
    $target = !empty($settings['target']) ? ' target="' . esc_attr($settings['target']) . '"' : '';
    
    return sprintf(
        '<a href="%1$s" class="%2$s"%3$s%4$s><span>%5$s</span></a>',
        esc_url($settings['url']),
        esc_attr($settings['class']),
        $target,
        $attr_html,
        esc_html($settings['text'])
    );
}

/**
 * Get menu items from ACF options
 * 
 * @return array Array of menu items with submenu
 */
function casino_get_menu_items() {
    $lang_slug = pll_current_language();
    $post_id = $lang_slug ? 'options_' . $lang_slug : 'option';
    $field_name = 'header_menu_' . $lang_slug;
    return get_field($field_name, $post_id) ?: [];
}

function casino_get_header_logo() {
    $lang_slug = pll_current_language();
    $post_id = $lang_slug ? 'options_' . $lang_slug : 'option';
    $field_name = 'header_logo_' . $lang_slug;
    $header_logo = get_field($field_name, $post_id);
    
    $logo = [
        'url' => get_template_directory_uri() . '/assets/public/img/logo.svg',
        'alt' => get_bloginfo('name')
    ];

    if (!empty($header_logo['url'])) {
        $logo['url'] = $header_logo['url'];
    }
    if (!empty($header_logo['alt'])) {
        $logo['alt'] = $header_logo['alt'];
    }
    return $logo;
}

function casino_get_footer_logo() {
    $lang_slug = pll_current_language();
    $post_id = $lang_slug ? 'options_' . $lang_slug : 'option';
    $field_name = 'footer_logo_' . $lang_slug;
    $footer_logo = get_field($field_name, $post_id);

    $logo = [
        'url' => get_template_directory_uri() . '/assets/public/img/logo.svg',
        'alt' => get_bloginfo('name')
    ];

    if (!empty($footer_logo['url'])) {
        $logo['url'] = $footer_logo['url'];
    }
    if (!empty($footer_logo['alt'])) {
        $logo['alt'] = $footer_logo['alt'];
    }
    return $logo;
}

function casino_get_social_networks() {
    $lang_slug = pll_current_language();
    $post_id = $lang_slug ? 'options_' . $lang_slug : 'option';
    $field_name = 'social_networks_' . $lang_slug;
    return get_field($field_name, $post_id) ?: [];
}

function casino_get_footer_menu_items() {
    $lang_slug = pll_current_language();
    $post_id = $lang_slug ? 'options_' . $lang_slug : 'option';
    $field_name = 'footer_menu_items_' . $lang_slug;
    return get_field($field_name, $post_id) ?: [];
}

function casino_get_footer_email($default = 'support@example.com') {
    $lang_slug = pll_current_language();
    $post_id = $lang_slug ? 'options_' . $lang_slug : 'option';
    $field_name = 'footer_email_' . $lang_slug;
    $email = get_field($field_name, $post_id);
    return !empty($email) ? $email : $default;
}

function casino_get_footer_text() {
    $lang_slug = pll_current_language();
    $post_id = $lang_slug ? 'options_' . $lang_slug : 'option';
    $field_name = 'footer_text_' . $lang_slug;
    return get_field($field_name, $post_id);
}

function casino_get_footer_images() {
    $lang_slug = pll_current_language();
    $post_id = $lang_slug ? 'options_' . $lang_slug : 'option';
    $field_name = 'footer_images_' . $lang_slug;
    return get_field($field_name, $post_id) ?: [];
}
